<?php

namespace Tests\Unit;

use App\Models\CompanyModel;
use CodeIgniter\Test\CIUnitTestCase;
use CodeIgniter\Test\DatabaseTestTrait;

/**
 * @internal
 */
final class CompaniesModelTest extends CIUnitTestCase
{
    use DatabaseTestTrait;

    protected $refresh = true;
    protected $seed = 'App\Database\Seeds\CompaniesSeeder';
    protected $seedOnce = false;
    protected $migrate = true;
    protected $migrateOnce = false;
    protected $namespace = 'App';

    public function testCanCreateCompany()
    {
        $model = new CompanyModel();

        $data = [
            'company_name'     => 'Test Company',
            'company_address'  => '123 Test Street, Test City',
            'area'             => 'Kachigam',
            'company_email'    => 'test@company.com',
            'hr_name'          => 'Test HR',
            'hr_number'        => '+919876543210',
            'total_workers'    => 50,
            'status'           => 'Active'
        ];

        $result = $model->save($data);

        $this->assertTrue($result);
        $this->seeInDatabase('companies', ['company_name' => 'Test Company']);
    }

    public function testCanRetrieveCompanies()
    {
        $model = new CompanyModel();
        $companies = $model->findAll();

        $this->assertCount(3, $companies);
    }

    public function testCanFindCompanyById()
    {
        $model = new CompanyModel();
        $company = $model->find(1);

        $this->assertIsArray($company);
        $this->assertEquals('ABC Corporation', $company['company_name']);
    }

    public function testCanUpdateCompany()
    {
        $model = new CompanyModel();

        $data = [
            'company_name' => 'Updated Company Name'
        ];

        $result = $model->update(1, $data);

        $this->assertTrue($result);
        $this->seeInDatabase('companies', ['id' => 1, 'company_name' => 'Updated Company Name']);
    }

    public function testCanDeleteCompany()
    {
        $model = new CompanyModel();
        $result = $model->update(1, ['status' => 'Inactive']);

        $this->assertTrue($result);
        $this->seeInDatabase('companies', ['id' => 1, 'status' => 'Inactive']);
    }

    public function testValidationFailsWithInvalidData()
    {
        $model = new CompanyModel();

        $data = [
            'company_name'     => '', // Required field
            'company_address'  => '', // Required field
            'area'             => 'Invalid Area', // Not in allowed list
            'company_email'    => 'invalid-email', // Invalid email format
            'hr_name'          => '', // Required field
            'hr_number'        => '123', // Invalid phone number format
            'total_workers'    => -5, // Must be >= 0
            'status'           => 'Invalid Status' // Not in allowed list
        ];

        $result = $model->save($data);

        $this->assertFalse($result);
        $this->assertNotEmpty($model->errors());
    }

    public function testGetAreasReturnsCorrectList()
    {
        $model = new CompanyModel();
        $areas = $model->getAreas();

        $this->assertIsArray($areas);
        $this->assertContains('Kachigam', $areas);
        $this->assertContains('Somnath', $areas);
        $this->assertCount(8, $areas);
    }
}