<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\SettingModel;
use CodeIgniter\HTTP\ResponseInterface;

class Settings extends BaseController
{
    protected $settingModel;

    public function __construct()
    {
        $this->settingModel = new SettingModel();
    }

    /**
     * Display the settings page
     *
     * @return string
     */
    public function index()
    {
        // Check user role for access control
        $role = session()->get('role');
        if (!$role) {
            return redirect()->to('/login');
        }

        // Get current settings
        $settings = $this->settingModel->getAll();

        return view('settings/index', [
            'title' => 'Settings',
            'settings' => $settings
        ]);
    }

    /**
     * Update general settings
     *
     * @return ResponseInterface
     */
    public function updateGeneral()
    {
        // Check user role for access control
        $role = session()->get('role');
        if (!$role) {
            return redirect()->to('/login');
        }

        // Only admin and superadmin can update settings
        if (!in_array($role, ['admin', 'superadmin'])) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Access denied');
        }

        // Get input data
        $appName = $this->request->getPost('app_name');
        $appTheme = $this->request->getPost('app_theme');
        $timezone = $this->request->getPost('timezone');

        // Save settings
        $this->settingModel->setValue('app_name', $appName, 'string', 'Application Name');
        $this->settingModel->setValue('app_theme', $appTheme, 'string', 'Application Theme');
        $this->settingModel->setValue('timezone', $timezone, 'string', 'Timezone');

        session()->setFlashdata('success', 'General settings updated successfully.');

        return redirect()->to('/settings');
    }

    /**
     * Update notification settings
     *
     * @return ResponseInterface
     */
    public function updateNotifications()
    {
        // Check user role for access control
        $role = session()->get('role');
        if (!$role) {
            return redirect()->to('/login');
        }

        // Only admin and superadmin can update settings
        if (!in_array($role, ['admin', 'superadmin'])) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Access denied');
        }

        // Get input data
        $emailNotifications = $this->request->getPost('email_notifications');
        $smsNotifications = $this->request->getPost('sms_notifications');
        $pushNotifications = $this->request->getPost('push_notifications');

        // Save settings
        $this->settingModel->setValue('email_notifications', $emailNotifications ? '1' : '0', 'boolean', 'Email Notifications');
        $this->settingModel->setValue('sms_notifications', $smsNotifications ? '1' : '0', 'boolean', 'SMS Notifications');
        $this->settingModel->setValue('push_notifications', $pushNotifications ? '1' : '0', 'boolean', 'Push Notifications');

        session()->setFlashdata('success', 'Notification settings updated successfully.');

        return redirect()->to('/settings');
    }

    /**
     * Update security settings
     *
     * @return ResponseInterface
     */
    public function updateSecurity()
    {
        // Check user role for access control
        $role = session()->get('role');
        if (!$role) {
            return redirect()->to('/login');
        }

        // Only admin and superadmin can update settings
        if (!in_array($role, ['admin', 'superadmin'])) {
            throw new \CodeIgniter\Exceptions\PageNotFoundException('Access denied');
        }

        // Get input data
        $twoFactorAuth = $this->request->getPost('two_factor_auth');
        $passwordExpiry = $this->request->getPost('password_expiry');

        // Save settings
        $this->settingModel->setValue('two_factor_auth', $twoFactorAuth ? '1' : '0', 'boolean', 'Two-Factor Authentication');
        $this->settingModel->setValue('password_expiry', $passwordExpiry, 'integer', 'Password Expiry (Days)');

        session()->setFlashdata('success', 'Security settings updated successfully.');

        return redirect()->to('/settings');
    }
}