# Companies Management Module

This module provides full CRUD functionality for managing companies in the CMS system.

## Features

1. **Add Company** - Form with validation for all required fields
2. **View Companies** - Paginated and searchable table with filter options
3. **Edit Company** - Form to update company information
4. **Delete Company** - Soft delete functionality (sets status to Inactive)
5. **View Details** - Dedicated page to view all company information
6. **Search & Filter** - Search by company name, HR name, or email; filter by area and status
7. **Export** - Export company list as CSV or PDF
8. **Role-based Access**:
   - Admin/Superadmin: Full access (Add, Edit, Delete, View, Export)
   - Other roles: Read-only access (View, Search, Filter)

## Form Fields

### Required Fields
1. Company Name (text, required, max 255 chars)
2. Company Address (textarea, required)
3. Area (dropdown, required) - Options: Kachigam, Dabhel, Bhimpore, Kadaiya, Ringanwada, Somnath, Bhenslore, Jani Vankad
4. Company Email ID (email, required, unique, valid email format)
5. HR Name (text, required)
6. HR Number (phone, required, validation for Indian numbers, allow +91 format)
7. Total Number of Workers (integer, required, >= 0)

### Optional Fields
8. Manager Name (text, optional)
9. Manager Number (phone, optional, same validation as HR)
10. GST / Tax Number (string, optional, 15 chars, alphanumeric validation)
11. Status (dropdown: Active / Inactive, default Active)
12. Notes (textarea, optional)

### Auto-generated Fields
13. Created At (auto timestamp)
14. Updated At (auto timestamp)

## Technical Implementation

### Database
- Table: `companies`
- Migration: `app/Database/Migrations/2025-09-25-000000_CreateCompaniesTable.php`

### Model
- `app/Models/CompanyModel.php`
- Includes validation rules and custom methods for data retrieval

### Controller
- `app/Controllers/Companies.php`
- Handles all CRUD operations and export functionality

### Views
- `app/Views/companies/list.php` - Companies listing with search/filter
- `app/Views/companies/create.php` - Add new company form
- `app/Views/companies/edit.php` - Edit company form
- `app/Views/companies/view.php` - Detailed company view

### Routes
Defined in `app/Config/Routes.php`:
```
$routes->group('companies', ['filter' => 'auth'], function ($routes) {
    $routes->get('/', 'Companies::index');
    $routes->get('create', 'Companies::create');
    $routes->post('store', 'Companies::store');
    $routes->get('edit/(:num)', 'Companies::edit/$1');
    $routes->post('update/(:num)', 'Companies::update/$1');
    $routes->get('delete/(:num)', 'Companies::delete/$1');
    $routes->get('view/(:num)', 'Companies::view/$1');
    $routes->get('export/csv', 'Companies::exportCsv');
    $routes->get('export/pdf', 'Companies::exportPdf');
});
```

## Validation Rules

The module implements comprehensive validation for all form fields:
- Required fields are validated for presence
- Email format and uniqueness validation
- Phone number validation for Indian format (+91 prefix optional)
- GST/Tax number validation (15-character alphanumeric)
- Area and Status dropdown validation
- Total workers must be a non-negative integer

## Export Functionality

### CSV Export
- Exports all visible columns in the companies table
- Includes all company information
- Generates downloadable CSV file

### PDF Export
- Generates a formatted PDF report
- Includes key company information
- Clean, professional layout

## Responsive Design

The module features a fully responsive design that works on:
- Desktop browsers
- Tablet devices
- Mobile devices

All forms and tables adapt to different screen sizes while maintaining usability.

## Access Control

The module implements role-based access control:
- Only authenticated users can access the module
- Admin and Superadmin roles have full CRUD access
- Other roles have read-only access
- Navigation menu items are only visible to authorized users

## Testing

Unit tests are available in `tests/unit/CompaniesModelTest.php` to verify:
- Company creation
- Data retrieval
- Updates and deletes
- Validation rules
- Helper methods

Note: Tests require the SQLite3 PHP extension to run.